//OHSAT GAMES TUTORIAL: MEGALAGA 6 BONUS: COLLISION & HUD
//There's a reference to Rolling in the OHSAT Tutorial for this lesson.
//It is your duty to make the reference to Weird Al and not Fred Durst or something equally terrible.   

//https://www.ohsat.com/tutorial/#mega-drive-tutorials 

#include <genesis.h>
#include <resources.h>

// Constants
#define LEFT_EDGE 0
#define RIGHT_EDGE 320

#define MAX_ENEMIES 6
#define MAX_BULLETS 3
#define MAX_FLY_ENEMIES 3
#define FLY_SPAWN_DELAY 180 //approx 3 seconds @ 60 FPS

#define ANIM_STRAIGHT 0
#define ANIM_MOVE 1
#define ANIM_DIE 7

typedef struct {
    int x, y, w, h;
    int velx, vely;
    int health;
    Sprite* sprite;
    char name[8];
} Entity;

// Globals
int offset = 0;
u16 enemiesLeft = 0;
int score = 0;
char hud_string[40];
bool flyEnemiesSpawned = FALSE;
u16 flySpawnTimer = 0; 

Entity player = {0}, player_2 = {0};
Entity enemies_top[MAX_ENEMIES];
Entity enemies_bottom[MAX_FLY_ENEMIES];
Entity bullets_p1[MAX_BULLETS], bullets_p2[MAX_BULLETS];
u16 bulletsOnScreenP1 = 0, bulletsOnScreenP2 = 0;

// Function declarations
void killEntity(Entity* e);
void reviveEntity(Entity* e);
void positionEnemies();
void positionPlayers();
void positionBullets();
void shootBullet(Entity* shooter, Entity* bulletArray, u16* bulletCount);
void handleCollisions();
void updateScoreDisplay();
void spawnFlyEnemies();
void myJoyHandler(u16 joy, u16 changed, u16 state);

int main() {
    JOY_init();
    JOY_setEventHandler(&myJoyHandler);

    SYS_disableInts();
    VDP_loadTileSet(background.tileset, 1, DMA);
    PAL_setPalette(PAL1, background.palette->data, DMA);
    PAL_setPalette(PAL2, background.palette->data, DMA);
    VDP_setScrollingMode(HSCROLL_PLANE, VSCROLL_PLANE);
    PAL_setColor(34, RGB24_TO_VDPCOLOR(0x0078f8));
    SYS_enableInts();

    SPR_init();

    // Init players
    player = (Entity){152, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship, 152, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P1"};
    player_2 = (Entity){100, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship1, 100, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P2"};

    // Fill background with stars
    for (int i = 0; i < 1280; i++) {
        int thex = i % 40;
        int they = i / 40;
        int val = (random() % 10) + 1;
        if (val > 3) val = 1;
        VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1, 0, 0, 0, val), thex, they);
    }

    // Initialize top enemies only
    enemiesLeft = 0;
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Sprite* spriteTop = SPR_addSprite(&ship, i * 48, 32, TILE_ATTR(PAL2, 0, TRUE, FALSE));
        if (spriteTop) {
            enemies_top[i] = (Entity){i * 48, 32, 16, 16, 1, 0, 1, spriteTop, ""};
            sprintf(enemies_top[i].name, "E1_%d", i);
            enemiesLeft++;
        } else {
            enemies_top[i].health = 0;
            enemies_top[i].sprite = NULL;
            VDP_drawText("Top sprite failed", 2, 12 + i);
        }
    }

    // Initialize fly enemies array but do not spawn
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        enemies_bottom[i].health = 0;
        enemies_bottom[i].sprite = NULL;
    }

    // Bullets
    for (int i = 0; i < MAX_BULLETS; i++) {
        bullets_p1[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
        bullets_p2[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet1, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
    }

    updateScoreDisplay();

    while (1) {
        VDP_setVerticalScroll(BG_B, offset -= 2);
        if (offset <= -256) offset = 0;

        positionPlayers();
        positionEnemies();
        positionBullets();
        handleCollisions();

        SPR_update();
        SYS_doVBlankProcess();
    }

    return 0;
}

// Input
void myJoyHandler(u16 joy, u16 changed, u16 state) {
    Entity* p = (joy == JOY_1) ? &player : &player_2;
    Entity* bullets = (joy == JOY_1) ? bullets_p1 : bullets_p2;
    u16* bulletCount = (joy == JOY_1) ? &bulletsOnScreenP1 : &bulletsOnScreenP2;

    if (state & BUTTON_RIGHT) {
        p->velx = 2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, TRUE);
    } else if (state & BUTTON_LEFT) {
        p->velx = -2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, FALSE);
    } else if ((changed & (BUTTON_LEFT | BUTTON_RIGHT))) {
        p->velx = 0;
        SPR_setAnim(p->sprite, ANIM_STRAIGHT);
    }

    if ((state & BUTTON_B) && (changed & BUTTON_B)) {
        shootBullet(p, bullets, bulletCount);
    }
}

// Movement
void positionPlayers() {
    Entity* ps[2] = { &player, &player_2 };
    for (int i = 0; i < 2; i++) {
        Entity* p = ps[i];
        p->x += p->velx;
        if (p->x < LEFT_EDGE) p->x = LEFT_EDGE;
        if (p->x + p->w > RIGHT_EDGE) p->x = RIGHT_EDGE - p->w;
        SPR_setPosition(p->sprite, p->x, p->y);
    }
}

void positionEnemies() {
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Entity* e1 = &enemies_top[i];
        if (e1->health > 0) {
            e1->x += e1->velx;
            if (e1->x < LEFT_EDGE || (e1->x + e1->w) > RIGHT_EDGE) e1->velx = -e1->velx;
            SPR_setPosition(e1->sprite, e1->x, e1->y);
        }
    }

for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
    Entity* f = &enemies_bottom[i];
    if (f->health > 0) {
        f->x += f->velx;

        // Bounce on edges instead of killing the enemy
        if (f->x < LEFT_EDGE) {
            f->x = LEFT_EDGE;
            f->velx = -f->velx;
        } else if (f->x + f->w > RIGHT_EDGE) {
            f->x = RIGHT_EDGE - f->w;
            f->velx = -f->velx;
        }

        SPR_setPosition(f->sprite, f->x, f->y);
    }
}
}

void shootBullet(Entity* shooter, Entity* bullets, u16* bulletCount) {
    if (*bulletCount < MAX_BULLETS) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b = &bullets[i];
            if (b->health == 0) {
                b->x = shooter->x + 4;
                b->y = shooter->y;
                b->vely = -3;
                reviveEntity(b);
                SPR_setPosition(b->sprite, b->x, b->y);
                (*bulletCount)++;
                break;
            }
        }
    }
}

void positionBullets() {
    for (int i = 0; i < MAX_BULLETS; i++) {
        Entity* b1 = &bullets_p1[i];
        Entity* b2 = &bullets_p2[i];
        if (b1->health > 0) {
            b1->y += b1->vely;
            if (b1->y + b1->h < 0) { killEntity(b1); bulletsOnScreenP1--; }
            else SPR_setPosition(b1->sprite, b1->x, b1->y);
        }
        if (b2->health > 0) {
            b2->y += b2->vely;
            if (b2->y + b2->h < 0) { killEntity(b2); bulletsOnScreenP2--; }
            else SPR_setPosition(b2->sprite, b2->x, b2->y);
        }
    }
}

int collideEntities(Entity* a, Entity* b) {
    return (a->x < b->x + b->w && a->x + a->w > b->x && a->y < b->y + b->h && a->y + a->h >= b->y);
}

void handleCollisions() {
    Entity* allEnemies[] = { enemies_top, enemies_bottom };
    int enemyCounts[] = { MAX_ENEMIES, MAX_FLY_ENEMIES };

    for (int g = 0; g < 2; g++) {
        Entity* group = allEnemies[g];
        for (int i = 0; i < enemyCounts[g]; i++) {
            Entity* e = &group[i];
            if (e->health == 0 || !e->sprite) continue;

            Entity* bulletSources[2] = { bullets_p1, bullets_p2 };
            u16* bulletCounts[2] = { &bulletsOnScreenP1, &bulletsOnScreenP2 };

            for (int j = 0; j < 2; j++) {
                for (int k = 0; k < MAX_BULLETS; k++) {
                    Entity* b = &bulletSources[j][k];
                    if (b->health > 0 && collideEntities(b, e)) {
                        killEntity(e);
                        if (g == 0) enemiesLeft--;
                        killEntity(b);
                        (*bulletCounts[j])--;
                        score += 10;
                        updateScoreDisplay();
                        goto next_enemy;
                    }
                }
            }
        next_enemy: ;
        }
    }

    if (!flyEnemiesSpawned && enemiesLeft == 0) {
        flySpawnTimer++;
        if (flySpawnTimer >= FLY_SPAWN_DELAY) {
            spawnFlyEnemies();
            flyEnemiesSpawned = TRUE;
        }
    }
}  

void spawnFlyEnemies() {
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        int startX = 40 + i * 60;  // spaced across screen width nicely
        int startY = 80 + i * 32;
        int vel = 2;  // initial velocity to the right
        Sprite* s = SPR_addSprite(&efly, startX, startY, TILE_ATTR(PAL2, 0, TRUE, FALSE));
        if (s) {
            enemies_bottom[i] = (Entity){startX, startY, 16, 16, vel, 0, 1, s, ""};
            sprintf(enemies_bottom[i].name, "Fly_%d", i);
        } else {
            enemies_bottom[i].health = 0;
            enemies_bottom[i].sprite = NULL;
        }
    }
}

void updateScoreDisplay() {
    sprintf(hud_string, "SCORE: %d  LEFT: %d", score, enemiesLeft);
    VDP_clearText(0, 0, 40);
    VDP_drawText(hud_string, 0, 0);
}

// Utility
void killEntity(Entity* e) {
    e->health = 0;
    if (e->sprite) SPR_setVisibility(e->sprite, HIDDEN);
}

void reviveEntity(Entity* e) {
    e->health = 1;
    if (e->sprite) SPR_setVisibility(e->sprite, VISIBLE);
}


/////////////////CHANGE LOG//////////////////

/*

Summary of Changes in Updated Code
Added Support for Two Players:

Introduced a second player entity (player_2) alongside the original player.

Created separate bullet arrays (bullets_p1 and bullets_p2) and bullet counters for each player.

Updated input handler (myJoyHandler) to handle input for both controllers (JOY_1 and JOY_2), allowing two players to move independently and shoot.

Split Enemies into Two Groups:

Top enemies (enemies_top) behave like the original enemies and move horizontally at the top of the screen.

Added a new group of “fly” enemies (enemies_bottom) that spawn later and have different movement (bounce instead of reversing velocity at edges).

Fly enemies spawn after a delay once all top enemies are defeated (controlled by flySpawnTimer and flyEnemiesSpawned).

Enemy Movement and Behavior Improvements:

Enemy movement logic refactored for both top and fly enemies.

Fly enemies bounce off screen edges instead of reversing velocity abruptly.

New Animation Support:

Added a new animation state ANIM_DIE (value 7), though the code doesn’t explicitly show its use yet—this sets the groundwork for enemy/player death animations.

Enhanced Bullet Management:

Bullets are now managed separately for each player, with each having its own maximum bullets on screen.

Bullet firing function shootBullet generalized to accept shooter and bullet arrays to support multiple players.

Improved Collision Handling:

Collision detection now checks bullets from both players against both enemy groups.

Uses nested loops and goto to skip to the next enemy after a collision is detected, preventing multiple hits on one enemy.

Decrements enemy counts and updates score accordingly.

Background and Sprite Initialization Enhancements:

Added safer sprite initialization with checks (e.g., testing if sprites were successfully created).

Background starfield generation logic remains similar but cleaned up slightly.

Code Style and Organization:

Used designated initializers and grouping of variables in structs for brevity.

Grouped players and enemies into arrays for cleaner iteration.

Added some clearer variable names (enemies_top, enemies_bottom, etc.).

Removed some unused variables and cleaned up formatting.

Additional Gameplay Features:

Implemented a timed delay before spawning the fly enemies group, adding pacing to the enemy waves.

Each player’s bullets are independent, allowing simultaneous multi-player gameplay.

*/

////////////////////NOTES////////////////////

/*

Code summary

🔧 Initialization
main()
Input setup:
JOY_init() and JOY_setEventHandler() configure gamepad input.

Video/sprite setup:

Loads background graphics and colors (VDP_loadTileSet, PAL_setPalette)

Enables horizontal and vertical scrolling

Initializes the sprite engine with SPR_init()

Players setup:

Creates two player entities (player, player_2) at the bottom of the screen with different sprites.

Background stars:

Randomly fills the background tilemap with stars to create a moving space effect.

Enemies:

Spawns top-row enemies (MAX_ENEMIES = 6). These start at the top and move side to side.

Initializes flying enemies (MAX_FLY_ENEMIES = 3) but does not spawn them yet.

Bullets:

Each player has a fixed array of bullet entities (3 max).

They’re initialized off-screen and hidden.

Main loop:

Scrolls the star background downward.

Calls all update functions:

Player movement

Enemy movement

Bullet movement

Collision handling

Refreshes sprite states and handles vertical blank timing.

🎮 Player Input Handling
myJoyHandler()
Detects left/right movement.

Moves the player horizontally.

Animates and flips the sprite accordingly.

Handles shooting:

If B is pressed and the player has bullets available, a bullet is spawned just above the player.

🚀 Game Mechanics
Movement Functions:
positionPlayers()

Updates each player’s horizontal position based on velocity.

Restricts players from moving off-screen.

Updates sprite positions.

positionEnemies()

Top enemies:

Move horizontally.

Reverse direction when they hit the screen edges.

Flying enemies:

Also move horizontally.

Bounce off screen edges (but are only active after spawning).

positionBullets()

Moves active bullets upward.

Destroys bullets if they leave the top of the screen.

Updates their sprite positions.

🎯 Combat & Collisions
handleCollisions()
Loops through both enemy groups (top and flying).

For each enemy:

Checks for collisions with all active bullets from both players.

If a bullet hits:

Both bullet and enemy are "killed"

Score increases

Updates HUD

Once all top enemies are destroyed, a timer starts.

After a short delay (FLY_SPAWN_DELAY), flying enemies are spawned.

👾 Enemy Spawning
spawnFlyEnemies()
After top enemies are gone, this function spawns new, horizontally-moving flying enemies.

They appear in spaced positions on screen and start moving right.

If a sprite fails to load, the enemy is disabled.

📺 HUD Updates
updateScoreDisplay()
Draws the current score and number of remaining top enemies at the top of the screen.

🔍 Utilities
killEntity() & reviveEntity()
Used to hide or show entities by setting their health and sprite visibility.

Helps manage bullet and enemy reuse without creating/destroying objects dynamically.


*/

/////////EXPERIMENTATION IDEAS///////////////

/*

May go back and modify the existing HUD to a more traditional HI SCORE system. We'll see what happens. 
Could always add a timer. 

We may revisit the HUD but I wanted to change the enemy sprite layout to avoid the miscount issue 
I was running into when using the NEMO sprite due to its size. Now, we have a new enemy sprite! 

New experimentation idea will involve the NEMO sprite but we'll revisit that down the line. 

*/

///////////ERROR HANDLING////////////////////
/*

*/